<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;

class CategoryController extends Controller
{
    public function index(Request $request): View
    {
        $query = Category::query();
        if ($q = $request->string('q')->toString()) {
            $query->where('name','like',"%{$q}%");
        }
        $categories = $query->latest('id')->paginate(15)->withQueryString();
        return view('admin.categories.index', compact('categories'));
    }

    public function create(): View
    {
        return view('admin.categories.create');
    }

    public function store(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'name' => ['required','string','max:50','unique:categories,name'],
            'description' => ['nullable','string'],
            'icon' => ['nullable','string','max:50'],
            'is_active' => ['boolean'],
        ]);
        Category::create($data);
        return redirect()->route('admin.categories.index')->with('success','تم إضافة التصنيف');
    }

    public function edit(Category $category): View
    {
        return view('admin.categories.edit', compact('category'));
    }

    public function update(Request $request, Category $category): RedirectResponse
    {
        $data = $request->validate([
            'name' => ['required','string','max:50',"unique:categories,name,{$category->id}"],
            'description' => ['nullable','string'],
            'icon' => ['nullable','string','max:50'],
            'is_active' => ['boolean'],
        ]);
        $category->update($data);
        return redirect()->route('admin.categories.index')->with('success','تم تحديث التصنيف');
    }

    public function destroy(Category $category): RedirectResponse
    {
        $category->delete();
        return redirect()->route('admin.categories.index')->with('success','تم حذف التصنيف');
    }
}


