<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    public function index(Request $request): View
    {
        $query = User::query();
        if ($type = $request->string('type')->toString()) {
            $query->where('user_type', $type);
        }
        if ($search = $request->string('q')->toString()) {
            $query->where(function ($q) use ($search) {
                $q->where('email', 'like', "%{$search}%")
                  ->orWhere('username', 'like', "%{$search}%")
                  ->orWhere('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%");
            });
        }
        $users = $query->latest('id')->paginate(15)->withQueryString();
        return view('admin.users.index', compact('users'));
    }

    public function create(): View
    {
        return view('admin.users.create');
    }

    public function store(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'username' => ['required','string','max:50','unique:users,username'],
            'email' => ['required','email','max:100','unique:users,email'],
            'password' => ['required','string','min:6'],
            'user_type' => ['required', Rule::in(['jobseeker','company','admin'])],
            'first_name' => ['required','string','max:50'],
            'last_name' => ['required','string','max:50'],
            'phone' => ['nullable','string','max:20'],
            'profile_image' => ['nullable','string','max:255'],
            'is_active' => ['boolean'],
            'email_verified' => ['boolean'],
        ]);
        $data['password'] = bcrypt($data['password']);
        User::create($data);
        return redirect()->route('admin.users.index')->with('success','تم إنشاء المستخدم');
    }

    public function edit(User $user): View
    {
        return view('admin.users.edit', compact('user'));
    }

    public function update(Request $request, User $user): RedirectResponse
    {
        $data = $request->validate([
            'username' => ['required','string','max:50', Rule::unique('users','username')->ignore($user->id)],
            'email' => ['required','email','max:100', Rule::unique('users','email')->ignore($user->id)],
            'password' => ['nullable','string','min:6'],
            'user_type' => ['required', Rule::in(['jobseeker','company','admin'])],
            'first_name' => ['required','string','max:50'],
            'last_name' => ['required','string','max:50'],
            'phone' => ['nullable','string','max:20'],
            'profile_image' => ['nullable','string','max:255'],
            'is_active' => ['boolean'],
            'email_verified' => ['boolean'],
        ]);
        if (!empty($data['password'])) {
            $data['password'] = bcrypt($data['password']);
        } else {
            unset($data['password']);
        }
        $user->update($data);
        return redirect()->route('admin.users.index')->with('success','تم تحديث المستخدم');
    }

    public function destroy(User $user): RedirectResponse
    {
        $user->delete();
        return redirect()->route('admin.users.index')->with('success','تم حذف المستخدم');
    }
}


