// Master-HR Website JavaScript
document.addEventListener('DOMContentLoaded', function() {

    // Initialize AOS (Animate On Scroll)
    AOS.init({
        duration: 1000,
        easing: 'ease-in-out',
        once: true,
        mirror: false
    });

    // Navbar scroll effect
    const navbar = document.getElementById('mainNav');
    window.addEventListener('scroll', function() {
        if (window.scrollY > 100) {
            navbar.style.background = 'rgba(37, 99, 235, 0.98)';
            navbar.style.boxShadow = '0 2px 20px rgba(0, 0, 0, 0.1)';
        } else {
            navbar.style.background = 'rgba(37, 99, 235, 0.95)';
            navbar.style.boxShadow = 'none';
        }
    });

    // Smooth scrolling for navigation links
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function(e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });

    // Form handling
    const jobseekerForm = document.getElementById('jobseekerForm');
    const companyForm = document.getElementById('companyForm');
    const contactForm = document.getElementById('contactForm');

    // Job Seeker Form
    if (jobseekerForm) {
        jobseekerForm.addEventListener('submit', function(e) {
            e.preventDefault();
            handleFormSubmission(this, 'jobseeker');
        });
    }

    // Company Form
    if (companyForm) {
        companyForm.addEventListener('submit', function(e) {
            e.preventDefault();
            handleFormSubmission(this, 'company');
        });
    }

    // Contact Form
    if (contactForm) {
        contactForm.addEventListener('submit', function(e) {
            e.preventDefault();
            handleFormSubmission(this, 'contact');
        });
    }

    // Form submission handler
    function handleFormSubmission(form, formType) {
        const formData = new FormData(form);
        const submitButton = form.querySelector('button[type="submit"]');
        const originalText = submitButton.innerHTML;

        // Show loading state
        submitButton.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>جاري الإرسال...';
        submitButton.disabled = true;

        // Simulate form submission (replace with actual API call)
        setTimeout(() => {
            showNotification('تم إرسال الطلب بنجاح! سنتواصل معك قريباً.', 'success');
            form.reset();

            // Reset button
            submitButton.innerHTML = originalText;
            submitButton.disabled = false;
        }, 2000);
    }

    // Notification system
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
                <span>${message}</span>
                <button class="notification-close">&times;</button>
            </div>
        `;

        // Add styles
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
            color: white;
            padding: 1rem 1.5rem;
            border-radius: 8px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
            z-index: 9999;
            transform: translateX(400px);
            transition: transform 0.3s ease;
            max-width: 400px;
        `;

        document.body.appendChild(notification);

        // Animate in
        setTimeout(() => {
            notification.style.transform = 'translateX(0)';
        }, 100);

        // Close button functionality
        const closeBtn = notification.querySelector('.notification-close');
        closeBtn.addEventListener('click', () => {
            notification.style.transform = 'translateX(400px)';
            setTimeout(() => {
                document.body.removeChild(notification);
            }, 300);
        });

        // Auto remove after 5 seconds
        setTimeout(() => {
            if (document.body.contains(notification)) {
                notification.style.transform = 'translateX(400px)';
                setTimeout(() => {
                    if (document.body.contains(notification)) {
                        document.body.removeChild(notification);
                    }
                }, 300);
            }
        }, 5000);
    }

    // Job application functionality
    document.querySelectorAll('.job-card .btn').forEach(button => {
        button.addEventListener('click', function() {
            const jobTitle = this.closest('.job-card').querySelector('h5').textContent;
            showJobApplicationModal(jobTitle);
        });
    });

    // Job application modal
    function showJobApplicationModal(jobTitle) {
        const modal = document.createElement('div');
        modal.className = 'job-application-modal';
        modal.innerHTML = `
            <div class="modal-overlay">
                <div class="modal-content">
                    <div class="modal-header">
                        <h4>تقدم لوظيفة: ${jobTitle}</h4>
                        <button class="modal-close">&times;</button>
                    </div>
                    <div class="modal-body">
                        <form id="jobApplicationForm">
                            <div class="mb-3">
                                <label for="applicantName" class="form-label">الاسم الكامل</label>
                                <input type="text" class="form-control" id="applicantName" required>
                            </div>
                            <div class="mb-3">
                                <label for="applicantEmail" class="form-label">البريد الإلكتروني</label>
                                <input type="email" class="form-control" id="applicantEmail" required>
                            </div>
                            <div class="mb-3">
                                <label for="applicantPhone" class="form-label">رقم الهاتف</label>
                                <input type="tel" class="form-control" id="applicantPhone" required>
                            </div>
                            <div class="mb-3">
                                <label for="coverLetter" class="form-label">رسالة التقديم</label>
                                <textarea class="form-control" id="coverLetter" rows="4" placeholder="اكتب رسالة قصيرة تشرح فيها لماذا أنت مناسب لهذه الوظيفة"></textarea>
                            </div>
                            <div class="mb-3">
                                <label for="resumeUpload" class="form-label">السيرة الذاتية</label>
                                <input type="file" class="form-control" id="resumeUpload" accept=".pdf,.doc,.docx" required>
                            </div>
                            <div class="text-center">
                                <button type="submit" class="btn btn-primary">إرسال الطلب</button>
                                <button type="button" class="btn btn-secondary ms-2" onclick="closeJobApplicationModal()">إلغاء</button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        `;

        // Add modal styles
        modal.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 10000;
            display: flex;
            align-items: center;
            justify-content: center;
        `;

        document.body.appendChild(modal);

        // Modal styles
        const style = document.createElement('style');
        style.textContent = `
            .job-application-modal .modal-overlay {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0, 0, 0, 0.5);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 10000;
            }
            
            .job-application-modal .modal-content {
                background: white;
                border-radius: 12px;
                max-width: 500px;
                width: 90%;
                max-height: 90vh;
                overflow-y: auto;
                box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
                animation: modalSlideIn 0.3s ease;
            }
            
            .job-application-modal .modal-header {
                padding: 1.5rem;
                border-bottom: 1px solid #e5e7eb;
                display: flex;
                justify-content: space-between;
                align-items: center;
            }
            
            .job-application-modal .modal-header h4 {
                margin: 0;
                color: #1f2937;
            }
            
            .job-application-modal .modal-close {
                background: none;
                border: none;
                font-size: 1.5rem;
                cursor: pointer;
                color: #6b7280;
                padding: 0;
                width: 30px;
                height: 30px;
                display: flex;
                align-items: center;
                justify-content: center;
                border-radius: 50%;
                transition: all 0.3s ease;
            }
            
            .job-application-modal .modal-close:hover {
                background: #f3f4f6;
                color: #1f2937;
            }
            
            .job-application-modal .modal-body {
                padding: 1.5rem;
            }
            
            @keyframes modalSlideIn {
                from {
                    opacity: 0;
                    transform: translateY(-50px) scale(0.9);
                }
                to {
                    opacity: 1;
                    transform: translateY(0) scale(1);
                }
            }
        `;

        document.head.appendChild(style);

        // Form submission
        const form = modal.querySelector('#jobApplicationForm');
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            showNotification('تم إرسال طلب التقديم بنجاح! سنتواصل معك قريباً.', 'success');
            closeJobApplicationModal();
        });

        // Close modal functionality
        const closeBtn = modal.querySelector('.modal-close');
        closeBtn.addEventListener('click', closeJobApplicationModal);

        // Close on overlay click
        modal.querySelector('.modal-overlay').addEventListener('click', function(e) {
            if (e.target === this) {
                closeJobApplicationModal();
            }
        });
    }

    // Close job application modal
    window.closeJobApplicationModal = function() {
        const modal = document.querySelector('.job-application-modal');
        if (modal) {
            modal.style.animation = 'modalSlideOut 0.3s ease';
            setTimeout(() => {
                if (document.body.contains(modal)) {
                    document.body.removeChild(modal);
                }
            }, 300);
        }
    };

    // Add slide out animation
    const slideOutStyle = document.createElement('style');
    slideOutStyle.textContent = `
        @keyframes modalSlideOut {
            from {
                opacity: 1;
                transform: translateY(0) scale(1);
            }
            to {
                opacity: 0;
                transform: translateY(-50px) scale(0.9);
            }
        }
    `;
    document.head.appendChild(slideOutStyle);

    // Newsletter subscription
    const newsletterForm = document.querySelector('.footer .input-group');
    if (newsletterForm) {
        const subscribeBtn = newsletterForm.querySelector('.btn');
        const emailInput = newsletterForm.querySelector('.form-control');

        subscribeBtn.addEventListener('click', function() {
            const email = emailInput.value.trim();
            if (email && isValidEmail(email)) {
                showNotification('تم الاشتراك في النشرة الإخبارية بنجاح!', 'success');
                emailInput.value = '';
            } else {
                showNotification('يرجى إدخال بريد إلكتروني صحيح.', 'error');
            }
        });
    }

    // Email validation
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }

    // Counter animation for statistics
    function animateCounters() {
        const counters = document.querySelectorAll('.counter');
        counters.forEach(counter => {
            const target = parseInt(counter.getAttribute('data-target'));
            const duration = 2000; // 2 seconds
            const increment = target / (duration / 16); // 60fps
            let current = 0;

            const updateCounter = () => {
                current += increment;
                if (current < target) {
                    counter.textContent = Math.floor(current);
                    requestAnimationFrame(updateCounter);
                } else {
                    counter.textContent = target;
                }
            };

            updateCounter();
        });
    }

    // Intersection Observer for counters
    const observerOptions = {
        threshold: 0.5,
        rootMargin: '0px 0px -100px 0px'
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                animateCounters();
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);

    // Observe counter section if it exists
    const counterSection = document.querySelector('.counters-section');
    if (counterSection) {
        observer.observe(counterSection);
    }

    // Parallax effect for hero section
    window.addEventListener('scroll', function() {
        const scrolled = window.pageYOffset;
        const heroSection = document.querySelector('.hero-section');
        if (heroSection) {
            const rate = scrolled * -0.5;
            heroSection.style.transform = `translateY(${rate}px)`;
        }
    });

    // Typing effect for hero title
    function typeWriter(element, text, speed = 100) {
        let i = 0;
        element.innerHTML = '';

        function type() {
            if (i < text.length) {
                element.innerHTML += text.charAt(i);
                i++;
                setTimeout(type, speed);
            }
        }

        type();
    }

    // Initialize typing effect if hero title exists
    const heroTitle = document.querySelector('.hero-content h1');
    if (heroTitle) {
        const originalText = heroTitle.textContent;
        setTimeout(() => {
            typeWriter(heroTitle, originalText, 50);
        }, 1000);
    }

    // Search functionality for jobs
    function initializeJobSearch() {
        const searchInput = document.querySelector('#jobSearch');
        if (searchInput) {
            searchInput.addEventListener('input', function() {
                const searchTerm = this.value.toLowerCase();
                const jobCards = document.querySelectorAll('.job-card');

                jobCards.forEach(card => {
                    const jobTitle = card.querySelector('h5').textContent.toLowerCase();
                    const companyName = card.querySelector('.company-info span').textContent.toLowerCase();
                    const tags = Array.from(card.querySelectorAll('.tag')).map(tag => tag.textContent.toLowerCase());

                    const isMatch = jobTitle.includes(searchTerm) ||
                        companyName.includes(searchTerm) ||
                        tags.some(tag => tag.includes(searchTerm));

                    card.style.display = isMatch ? 'block' : 'none';
                });
            });
        }
    }

    // Initialize job search
    initializeJobSearch();

    // Filter jobs by category
    function initializeJobFilters() {
        const filterButtons = document.querySelectorAll('.job-filter');
        if (filterButtons.length > 0) {
            filterButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const category = this.getAttribute('data-category');

                    // Remove active class from all buttons
                    filterButtons.forEach(btn => btn.classList.remove('active'));
                    // Add active class to clicked button
                    this.classList.add('active');

                    // Filter jobs
                    const jobCards = document.querySelectorAll('.job-card');
                    jobCards.forEach(card => {
                        if (category === 'all' || card.getAttribute('data-category') === category) {
                            card.style.display = 'block';
                            card.style.animation = 'fadeInUp 0.5s ease';
                        } else {
                            card.style.display = 'none';
                        }
                    });
                });
            });
        }
    }

    // Initialize job filters
    initializeJobFilters();

    // Back to top button
    const backToTopBtn = document.createElement('button');
    backToTopBtn.innerHTML = '<i class="fas fa-arrow-up"></i>';
    backToTopBtn.className = 'back-to-top';
    backToTopBtn.style.cssText = `
        position: fixed;
        bottom: 30px;
        right: 30px;
        width: 50px;
        height: 50px;
        background: var(--primary-color);
        color: white;
        border: none;
        border-radius: 50%;
        cursor: pointer;
        display: none;
        align-items: center;
        justify-content: center;
        font-size: 1.2rem;
        box-shadow: 0 5px 15px rgba(37, 99, 235, 0.3);
        transition: all 0.3s ease;
        z-index: 1000;
    `;

    document.body.appendChild(backToTopBtn);

    // Show/hide back to top button
    window.addEventListener('scroll', function() {
        if (window.pageYOffset > 300) {
            backToTopBtn.style.display = 'flex';
        } else {
            backToTopBtn.style.display = 'none';
        }
    });

    // Back to top functionality
    backToTopBtn.addEventListener('click', function() {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    });

    // Hover effects for back to top button
    backToTopBtn.addEventListener('mouseenter', function() {
        this.style.transform = 'translateY(-3px)';
        this.style.boxShadow = '0 8px 25px rgba(37, 99, 235, 0.4)';
    });

    backToTopBtn.addEventListener('mouseleave', function() {
        this.style.transform = 'translateY(0)';
        this.style.boxShadow = '0 5px 15px rgba(37, 99, 235, 0.3)';
    });

    // Loading animation for page elements
    function initializeLoadingAnimations() {
        const elements = document.querySelectorAll('.feature-card, .job-card, .company-card');
        elements.forEach((element, index) => {
            element.style.opacity = '0';
            element.style.transform = 'translateY(30px)';

            setTimeout(() => {
                element.style.transition = 'all 0.6s ease';
                element.style.opacity = '1';
                element.style.transform = 'translateY(0)';
            }, index * 100);
        });
    }

    // Initialize loading animations
    setTimeout(initializeLoadingAnimations, 500);

    // Mobile menu toggle enhancement
    const navbarToggler = document.querySelector('.navbar-toggler');
    const navbarCollapse = document.querySelector('.navbar-collapse');

    if (navbarToggler && navbarCollapse) {
        navbarToggler.addEventListener('click', function() {
            navbarCollapse.classList.toggle('show');

            // Add animation to mobile menu items
            const navItems = navbarCollapse.querySelectorAll('.nav-item');
            navItems.forEach((item, index) => {
                if (navbarCollapse.classList.contains('show')) {
                    item.style.animation = `slideInRight 0.3s ease ${index * 0.1}s both`;
                } else {
                    item.style.animation = 'none';
                }
            });
        });
    }

    // Close mobile menu when clicking on a link
    const mobileNavLinks = document.querySelectorAll('.navbar-nav .nav-link');
    mobileNavLinks.forEach(link => {
        link.addEventListener('click', function() {
            if (window.innerWidth < 992) {
                navbarCollapse.classList.remove('show');
            }
        });
    });

    // Performance optimization: Lazy loading for images
    if ('IntersectionObserver' in window) {
        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    img.src = img.dataset.src;
                    img.classList.remove('lazy');
                    imageObserver.unobserve(img);
                }
            });
        });

        document.querySelectorAll('img[data-src]').forEach(img => {
            imageObserver.observe(img);
        });
    }

    // Console welcome message
    console.log(`
    🚀 Welcome to Master-HR Website!
    
    ✨ Features:
    - Modern & Responsive Design
    - Interactive Forms
    - Smooth Animations
    - Job Search & Filtering
    - Mobile-First Approach
    
    🎨 Built with:
    - HTML5, CSS3, JavaScript
    - Bootstrap 5
    - Font Awesome Icons
    - AOS Animation Library
    
    📱 For support: info@master-hr.com
    `);

});

// Events Section Functionality
let eventScene, eventCamera, eventRenderer, eventCube;
let eventAnimationId;
let eventMouseX = 0,
    eventMouseY = 0;

// Initialize 3D Event Container with enhanced features
function initEvent3D() {
    const container = document.getElementById('event3dContainer');
    if (!container) return;

    // Scene setup
    eventScene = new THREE.Scene();

    // Camera setup with better positioning
    eventCamera = new THREE.PerspectiveCamera(75, container.clientWidth / container.clientHeight, 0.1, 1000);
    eventCamera.position.set(0, 0, 8);

    // Renderer setup with enhanced quality
    eventRenderer = new THREE.WebGLRenderer({
        alpha: true,
        antialias: true,
        powerPreference: "high-performance"
    });
    eventRenderer.setSize(container.clientWidth, container.clientHeight);
    eventRenderer.setClearColor(0x000000, 0);
    eventRenderer.shadowMap.enabled = true;
    eventRenderer.shadowMap.type = THREE.PCFSoftShadowMap;
    container.appendChild(eventRenderer.domElement);

    // Create enhanced floating geometric shapes
    createEnhancedEventShapes();

    // Add enhanced lighting
    const ambientLight = new THREE.AmbientLight(0xffffff, 0.7);
    eventScene.add(ambientLight);

    const directionalLight = new THREE.DirectionalLight(0xffffff, 1);
    directionalLight.position.set(15, 15, 10);
    directionalLight.castShadow = true;
    directionalLight.shadow.mapSize.width = 2048;
    directionalLight.shadow.mapSize.height = 2048;
    eventScene.add(directionalLight);

    // Add point lights for dynamic lighting
    const pointLight1 = new THREE.PointLight(0x667eea, 1, 20);
    pointLight1.position.set(-5, 5, 5);
    eventScene.add(pointLight1);

    const pointLight2 = new THREE.PointLight(0x764ba2, 1, 20);
    pointLight2.position.set(5, -5, 5);
    eventScene.add(pointLight2);

    // Start enhanced animation
    animateEnhancedEvent3D();

    // Mark as loaded with fade effect
    setTimeout(() => {
        container.classList.add('loaded');
        container.style.opacity = '1';
    }, 1500);

    // Handle resize with debouncing
    let resizeTimeout;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(onEventWindowResize, 250);
    });

    // Add mouse interaction for 3D scene
    container.addEventListener('mousemove', onEventMouseMove);
    container.addEventListener('touchmove', onEventTouchMove);

    // Add loading animation
    addLoadingAnimation(container);
}

// Create enhanced floating geometric shapes with more variety
function createEnhancedEventShapes() {
    const geometries = [
        new THREE.BoxGeometry(1, 1, 1),
        new THREE.SphereGeometry(0.8, 32, 32),
        new THREE.ConeGeometry(0.6, 1.2, 32),
        new THREE.TorusGeometry(0.6, 0.25, 16, 100),
        new THREE.OctahedronGeometry(0.7),
        new THREE.TetrahedronGeometry(0.8)
    ];

    const materials = [
        new THREE.MeshPhongMaterial({
            color: 0x667eea,
            transparent: true,
            opacity: 0.9,
            shininess: 100
        }),
        new THREE.MeshPhongMaterial({
            color: 0x764ba2,
            transparent: true,
            opacity: 0.9,
            shininess: 100
        }),
        new THREE.MeshPhongMaterial({
            color: 0x4facfe,
            transparent: true,
            opacity: 0.9,
            shininess: 100
        }),
        new THREE.MeshPhongMaterial({
            color: 0x00f2fe,
            transparent: true,
            opacity: 0.9,
            shininess: 100
        }),
        new THREE.MeshPhongMaterial({
            color: 0xf093fb,
            transparent: true,
            opacity: 0.9,
            shininess: 100
        }),
        new THREE.MeshPhongMaterial({
            color: 0xf5576c,
            transparent: true,
            opacity: 0.9,
            shininess: 100
        })
    ];

    // Create more shapes for richer scene
    for (let i = 0; i < 12; i++) {
        const geometry = geometries[i % geometries.length];
        const material = materials[i % materials.length].clone();
        const mesh = new THREE.Mesh(geometry, material);

        // Better positioning
        mesh.position.set(
            (Math.random() - 0.5) * 10,
            (Math.random() - 0.5) * 10,
            (Math.random() - 0.5) * 10
        );

        mesh.rotation.set(
            Math.random() * Math.PI,
            Math.random() * Math.PI,
            Math.random() * Math.PI
        );

        // Enhanced animation properties
        mesh.userData = {
            rotationSpeed: {
                x: (Math.random() - 0.5) * 0.03,
                y: (Math.random() - 0.5) * 0.03,
                z: (Math.random() - 0.5) * 0.03
            },
            floatSpeed: Math.random() * 0.008 + 0.003,
            floatOffset: Math.random() * Math.PI * 2,
            floatAmplitude: Math.random() * 0.5 + 0.3,
            scaleSpeed: Math.random() * 0.01 + 0.005,
            originalScale: 1
        };

        // Add shadows
        mesh.castShadow = true;
        mesh.receiveShadow = true;

        eventScene.add(mesh);
    }

    // Add particle system for extra visual appeal
    createParticleSystem();
}

// Create particle system for enhanced visual effects
function createParticleSystem() {
    const particleCount = 50;
    const particles = new THREE.BufferGeometry();
    const positions = new Float32Array(particleCount * 3);
    const colors = new Float32Array(particleCount * 3);

    for (let i = 0; i < particleCount; i++) {
        positions[i * 3] = (Math.random() - 0.5) * 20;
        positions[i * 3 + 1] = (Math.random() - 0.5) * 20;
        positions[i * 3 + 2] = (Math.random() - 0.5) * 20;

        colors[i * 3] = Math.random() * 0.5 + 0.5;
        colors[i * 3 + 1] = Math.random() * 0.5 + 0.5;
        colors[i * 3 + 2] = Math.random() * 0.5 + 0.5;
    }

    particles.setAttribute('position', new THREE.BufferAttribute(positions, 3));
    particles.setAttribute('color', new THREE.BufferAttribute(colors, 3));

    const particleMaterial = new THREE.PointsMaterial({
        size: 0.1,
        vertexColors: true,
        transparent: true,
        opacity: 0.6
    });

    const particleSystem = new THREE.Points(particles, particleMaterial);
    eventScene.add(particleSystem);

    // Store for animation
    eventScene.userData.particleSystem = particleSystem;
}

// Enhanced 3D animation with mouse interaction
function animateEnhancedEvent3D() {
    eventAnimationId = requestAnimationFrame(animateEnhancedEvent3D);

    // Rotate and float all meshes with enhanced effects
    eventScene.children.forEach(child => {
        if (child.isMesh) {
            // Enhanced rotation
            child.rotation.x += child.userData.rotationSpeed.x;
            child.rotation.y += child.userData.rotationSpeed.y;
            child.rotation.z += child.userData.rotationSpeed.z;

            // Enhanced floating motion
            const time = Date.now() * child.userData.floatSpeed;
            const floatY = Math.sin(time + child.userData.floatOffset) * child.userData.floatAmplitude;
            child.position.y += floatY * 0.01;

            // Gentle rotation around Y axis
            child.rotation.y += 0.003;

            // Scale animation
            const scale = 1 + Math.sin(time * child.userData.scaleSpeed) * 0.1;
            child.scale.setScalar(scale);

            // Mouse interaction effect
            const distanceFromMouse = Math.sqrt(
                Math.pow(child.position.x - eventMouseX * 0.1, 2) +
                Math.pow(child.position.y - eventMouseY * 0.1, 2)
            );

            if (distanceFromMouse < 3) {
                child.material.opacity = 1;
                child.scale.setScalar(scale * 1.2);
            } else {
                child.material.opacity = 0.9;
            }
        }
    });

    // Animate particle system
    if (eventScene.userData.particleSystem) {
        const particles = eventScene.userData.particleSystem.geometry.attributes.position.array;
        for (let i = 0; i < particles.length; i += 3) {
            particles[i + 1] += Math.sin(Date.now() * 0.001 + i) * 0.001;
        }
        eventScene.userData.particleSystem.geometry.attributes.position.needsUpdate = true;
    }

    // Camera movement based on mouse position
    eventCamera.position.x += (eventMouseX * 0.01 - eventCamera.position.x) * 0.05;
    eventCamera.position.y += (-eventMouseY * 0.01 - eventCamera.position.y) * 0.05;
    eventCamera.lookAt(0, 0, 0);

    // Render scene
    if (eventRenderer && eventScene && eventCamera) {
        eventRenderer.render(eventScene, eventCamera);
    }
}

// Mouse interaction for 3D scene
function onEventMouseMove(event) {
    const container = document.getElementById('event3dContainer');
    if (!container) return;

    const rect = container.getBoundingClientRect();
    eventMouseX = ((event.clientX - rect.left) / container.clientWidth) * 2 - 1;
    eventMouseY = -((event.clientY - rect.top) / container.clientHeight) * 2 + 1;
}

// Touch interaction for mobile
function onEventTouchMove(event) {
    event.preventDefault();
    const container = document.getElementById('event3dContainer');
    if (!container) return;

    const rect = container.getBoundingClientRect();
    const touch = event.touches[0];
    eventMouseX = ((touch.clientX - rect.left) / container.clientWidth) * 2 - 1;
    eventMouseY = -((touch.clientY - rect.top) / container.clientHeight) * 2 + 1;
}

// Add loading animation with progress bar
function addLoadingAnimation(container) {
    const loadingOverlay = document.createElement('div');
    loadingOverlay.className = 'loading-overlay';
    loadingOverlay.innerHTML = `
        <div class="loading-content">
            <div class="loading-spinner"></div>
            <div class="loading-text">جاري تحميل العرض ثلاثي الأبعاد...</div>
            <div class="loading-progress">
                <div class="loading-bar"></div>
            </div>
        </div>
    `;

    container.appendChild(loadingOverlay);

    // Simulate loading progress
    let progress = 0;
    const progressBar = loadingOverlay.querySelector('.loading-bar');
    const interval = setInterval(() => {
        progress += Math.random() * 15;
        if (progress >= 100) {
            progress = 100;
            clearInterval(interval);
            setTimeout(() => {
                loadingOverlay.style.opacity = '0';
                setTimeout(() => {
                    if (loadingOverlay.parentNode) {
                        loadingOverlay.parentNode.removeChild(loadingOverlay);
                    }
                }, 500);
            }, 500);
        }
        progressBar.style.width = progress + '%';
    }, 100);
}

// Enhanced event subscription with better UX
function subscribeToEvent(eventId) {
    const eventData = getEventData(eventId);

    // Add loading state to button
    const button = event.target;
    const originalText = button.innerHTML;
    button.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>جاري التحميل...';
    button.disabled = true;

    // Simulate loading delay for better UX
    setTimeout(() => {
        showEventSubscriptionModal(eventData);
        button.innerHTML = originalText;
        button.disabled = false;
    }, 800);
}

// Get event data by ID
function getEventData(eventId) {
    const events = {
        'summit-2024': {
            title: 'قمة التوظيف الرقمي 2024',
            date: '15-17 مارس 2024',
            location: 'فندق الرياض الدولي - الرياض',
            price: 'مجاني',
            capacity: '500+ مشارك'
        },
        'workshop-career': {
            title: 'ورشة التطوير المهني',
            date: '22 فبراير 2024',
            location: 'مركز التدريب المهني - جدة',
            price: '200 ريال',
            capacity: '50 مشارك'
        },
        'seminar-ai-hr': {
            title: 'ندوة التكنولوجيا في التوظيف',
            date: '08 مارس 2024',
            location: 'جامعة الملك فهد - الدمام',
            price: 'مجاني',
            capacity: '200 مشارك'
        },
        'job-fair-spring': {
            title: 'معرض الوظائف الربيعي',
            date: '30 مارس 2024',
            location: 'مركز المعارض - الرياض',
            price: '50 ريال',
            capacity: '1000+ مشارك'
        }
    };

    return events[eventId] || {};
}

// Show event subscription modal
function showEventSubscriptionModal(eventData) {
    const modalHtml = `
        <div class="modal fade" id="eventSubscriptionModal" tabindex="-1">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">اشتراك في الفعالية</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="event-summary mb-4">
                            <h6 class="text-primary">${eventData.title}</h6>
                            <div class="row">
                                <div class="col-md-6">
                                    <p><i class="fas fa-calendar-alt text-muted"></i> ${eventData.date}</p>
                                    <p><i class="fas fa-map-marker-alt text-muted"></i> ${eventData.location}</p>
                                </div>
                                <div class="col-md-6">
                                    <p><i class="fas fa-tag text-muted"></i> ${eventData.price}</p>
                                    <p><i class="fas fa-users text-muted"></i> ${eventData.capacity}</p>
                                </div>
                            </div>
                        </div>
                        
                        <form id="eventSubscriptionForm">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">الاسم الأول</label>
                                    <input type="text" class="form-control" required>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">الاسم الأخير</label>
                                    <input type="text" class="form-control" required>
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">البريد الإلكتروني</label>
                                    <input type="email" class="form-control" required>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">رقم الهاتف</label>
                                    <input type="tel" class="form-control" required>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">المؤهل العلمي</label>
                                <select class="form-select" required>
                                    <option value="">اختر المؤهل</option>
                                    <option value="bachelor">بكالوريوس</option>
                                    <option value="master">ماجستير</option>
                                    <option value="phd">دكتوراه</option>
                                    <option value="diploma">دبلوم</option>
                                    <option value="other">أخرى</option>
                                </select>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">المجال المهني</label>
                                <input type="text" class="form-control" placeholder="مثال: تكنولوجيا المعلومات، تسويق، موارد بشرية">
                            </div>
                            <div class="mb-3">
                                <label class="form-label">سبب الاهتمام بالفعالية</label>
                                <textarea class="form-control" rows="3" placeholder="اكتب لماذا تريد حضور هذه الفعالية"></textarea>
                            </div>
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" id="newsletterSubscription">
                                <label class="form-check-label" for="newsletterSubscription">
                                    أريد الاشتراك في النشرة الإخبارية للحصول على آخر الفعاليات
                                </label>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                        <button type="button" class="btn btn-primary" onclick="submitEventSubscription()">
                            <i class="fas fa-check me-2"></i>
                            تأكيد الاشتراك
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;

    // Remove existing modal if any
    const existingModal = document.getElementById('eventSubscriptionModal');
    if (existingModal) {
        existingModal.remove();
    }

    // Add modal to body
    document.body.insertAdjacentHTML('beforeend', modalHtml);

    // Show modal
    const modal = new bootstrap.Modal(document.getElementById('eventSubscriptionModal'));
    modal.show();
}

// Submit event subscription
function submitEventSubscription() {
    const form = document.getElementById('eventSubscriptionForm');
    if (!form.checkValidity()) {
        form.reportValidity();
        return;
    }

    // Show loading state
    const submitBtn = form.querySelector('.btn-primary');
    const originalText = submitBtn.innerHTML;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>جاري الإرسال...';
    submitBtn.disabled = true;

    // Simulate API call
    setTimeout(() => {
        // Show success message
        showNotification('تم الاشتراك في الفعالية بنجاح! سنتواصل معك قريباً', 'success');

        // Close modal
        const modal = bootstrap.Modal.getInstance(document.getElementById('eventSubscriptionModal'));
        modal.hide();

        // Reset form
        form.reset();

        // Reset button
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    }, 2000);
}

// View event details
function viewEventDetails(eventId) {
    const eventData = getEventData(eventId);

    // Show detailed information modal
    showEventDetailsModal(eventData);
}

// Enhanced event details modal with better animations
function showEventDetailsModal(eventData) {
    const modalHtml = `
        <div class="modal fade" id="eventDetailsModal" tabindex="-1">
            <div class="modal-dialog modal-xl">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">
                            <i class="fas fa-calendar-alt text-primary me-2"></i>
                            تفاصيل الفعالية
                        </h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="event-details-content">
                            <h4 class="text-primary mb-4">${eventData.title}</h4>
                            
                            <div class="row">
                                <div class="col-md-8">
                                    <div class="event-description-detailed">
                                        <h6><i class="fas fa-info-circle text-info me-2"></i>وصف الفعالية</h6>
                                        <p>هذه الفعالية المميزة تقدم فرصة فريدة للتعلم والتطوير المهني. ستتعرف على أحدث الاتجاهات في مجال عملك وتلتقي بخبراء الصناعة.</p>
                                        
                                        <h6><i class="fas fa-list-ul text-success me-2"></i>المحاور الرئيسية</h6>
                                        <ul>
                                            <li>أحدث التقنيات والاتجاهات في المجال</li>
                                            <li>استراتيجيات التطوير المهني</li>
                                            <li>فرص التواصل مع الخبراء</li>
                                            <li>ورش عمل تفاعلية</li>
                                        </ul>
                                        
                                        <h6><i class="fas fa-users text-warning me-2"></i>الفئة المستهدفة</h6>
                                        <p>المهنيون في جميع المستويات، من المبتدئين إلى ذوي الخبرة العالية.</p>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="event-info-sidebar">
                                        <div class="info-card">
                                            <h6><i class="fas fa-calendar-alt text-primary"></i> التاريخ والوقت</h6>
                                            <p>${eventData.date}</p>
                                        </div>
                                        <div class="info-card">
                                            <h6><i class="fas fa-map-marker-alt text-success"></i> الموقع</h6>
                                            <p>${eventData.location}</p>
                                        </div>
                                        <div class="info-card">
                                            <h6><i class="fas fa-tag text-warning"></i> السعر</h6>
                                            <p>${eventData.price}</p>
                                        </div>
                                        <div class="info-card">
                                            <h6><i class="fas fa-users text-info"></i> السعة</h6>
                                            <p>${eventData.capacity}</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-times me-2"></i>إغلاق
                        </button>
                        <button type="button" class="btn btn-primary" onclick="subscribeToEvent('${eventId}')">
                            <i class="fas fa-ticket-alt me-2"></i>
                            اشترك الآن
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;

    // Remove existing modal if any
    const existingModal = document.getElementById('eventDetailsModal');
    if (existingModal) {
        existingModal.remove();
    }

    // Add modal to body
    document.body.insertAdjacentHTML('beforeend', modalHtml);

    // Show modal with enhanced animation
    const modal = new bootstrap.Modal(document.getElementById('eventDetailsModal'));
    modal.show();

    // Add entrance animation to modal content
    setTimeout(() => {
        const modalContent = document.querySelector('#eventDetailsModal .modal-content');
        modalContent.style.animation = 'modalSlideIn 0.5s ease forwards';
    }, 100);
}

// Enhanced organize event functionality
function organizeEvent() {
    // Show enhanced notification
    showNotification('سيتم التواصل معك قريباً لتنظيم الفعالية. سنرسل لك تفاصيل أكثر عبر البريد الإلكتروني.', 'info');

    // Add success animation to button
    const button = event.target;
    button.innerHTML = '<i class="fas fa-check me-2"></i>تم الإرسال!';
    button.classList.add('btn-success');

    setTimeout(() => {
        button.innerHTML = '<i class="fas fa-calendar-plus me-2"></i>تنظيم فعالية';
        button.classList.remove('btn-success');
    }, 3000);
}

// Event category filtering
function initializeEventCategories() {
    const categoryBtns = document.querySelectorAll('.category-btn');

    categoryBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            // Remove active class from all buttons
            categoryBtns.forEach(b => b.classList.remove('active'));

            // Add active class to clicked button
            this.classList.add('active');

            // Get category
            const category = this.dataset.category;

            // Filter events (placeholder for future implementation)
            filterEventsByCategory(category);
        });
    });
}

// Enhanced event category filtering with smooth animations
function filterEventsByCategory(category) {
    const eventCards = document.querySelectorAll('.event-card');
    const featuredCard = document.querySelector('.featured-event-card');

    // Add loading state
    showNotification(`جاري تصفية الفعاليات في فئة: ${category === 'all' ? 'جميع الفعاليات' : category}`, 'info');

    // Smooth transition for all cards
    eventCards.forEach((card, index) => {
        setTimeout(() => {
            if (category === 'all' || card.dataset.category === category) {
                card.style.display = 'block';
                card.style.animation = 'fadeInUp 0.6s ease forwards';
            } else {
                card.style.animation = 'fadeOutDown 0.4s ease forwards';
                setTimeout(() => {
                    card.style.display = 'none';
                }, 400);
            }
        }, index * 100);
    });

    // Show/hide featured card based on category
    if (category === 'all' || category === 'featured') {
        if (featuredCard) {
            featuredCard.style.display = 'block';
            featuredCard.style.animation = 'fadeInUp 0.8s ease forwards';
        }
    } else {
        if (featuredCard) {
            featuredCard.style.animation = 'fadeOutDown 0.6s ease forwards';
            setTimeout(() => {
                featuredCard.style.display = 'none';
            }, 600);
        }
    }
}

// Initialize events section with enhanced features
document.addEventListener('DOMContentLoaded', function() {
    // Initialize 3D animations if Three.js is available
    if (typeof THREE !== 'undefined') {
        setTimeout(initEvent3D, 1000);
    } else {
        console.warn('Three.js not available. 3D animations will be disabled.');
        // Fallback: show static content
        const container = document.getElementById('event3dContainer');
        if (container) {
            container.innerHTML = `
                <div class="static-event-preview">
                    <i class="fas fa-calendar-alt fa-3x text-white mb-3"></i>
                    <h4 class="text-white">قمة التوظيف الرقمي 2024</h4>
                    <p class="text-white-50">أكبر تجمع للمهنيين وقادة الموارد البشرية</p>
                </div>
            `;
            container.classList.add('loaded');
        }
    }

    // Initialize event categories with enhanced functionality
    initializeEventCategories();

    // Add scroll animations for event cards
    addScrollAnimations();

    // Initialize event search functionality
    initializeEventSearch();
});

// Add scroll animations for better mobile experience
function addScrollAnimations() {
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.animation = 'fadeInUp 0.8s ease forwards';
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);

    // Observe event cards
    document.querySelectorAll('.event-card').forEach(card => {
        observer.observe(card);
    });
}

// Initialize event search functionality
function initializeEventSearch() {
    const searchInput = document.createElement('input');
    searchInput.type = 'text';
    searchInput.placeholder = 'ابحث في الفعاليات...';
    searchInput.className = 'form-control event-search-input';
    searchInput.style.cssText = `
        max-width: 400px;
        margin: 0 auto 2rem auto;
        border-radius: 25px;
        padding: 0.8rem 1.5rem;
        border: 2px solid #e9ecef;
        transition: all 0.3s ease;
    `;

    // Insert search input before event categories
    const categoriesSection = document.querySelector('.event-categories');
    if (categoriesSection) {
        categoriesSection.parentNode.insertBefore(searchInput, categoriesSection);

        // Add search functionality
        searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const eventCards = document.querySelectorAll('.event-card');
            const featuredCard = document.querySelector('.featured-event-card');

            eventCards.forEach(card => {
                const title = card.querySelector('h5').textContent.toLowerCase();
                const description = card.querySelector('p').textContent.toLowerCase();

                if (title.includes(searchTerm) || description.includes(searchTerm)) {
                    card.style.display = 'block';
                    card.style.animation = 'fadeInUp 0.5s ease forwards';
                } else {
                    card.style.display = 'none';
                }
            });

            // Show/hide featured card based on search
            if (featuredCard) {
                const featuredTitle = featuredCard.querySelector('.event-title').textContent.toLowerCase();
                const featuredDesc = featuredCard.querySelector('.event-description').textContent.toLowerCase();

                if (featuredTitle.includes(searchTerm) || featuredDesc.includes(searchTerm)) {
                    featuredCard.style.display = 'block';
                } else {
                    featuredCard.style.display = 'none';
                }
            }
        });
    }
}

// Cleanup 3D animations when page is unloaded
window.addEventListener('beforeunload', function() {
    if (eventAnimationId) {
        cancelAnimationFrame(eventAnimationId);
    }

    if (eventRenderer) {
        eventRenderer.dispose();
    }

    // Remove event listeners
    const container = document.getElementById('event3dContainer');
    if (container) {
        container.removeEventListener('mousemove', onEventMouseMove);
        container.removeEventListener('touchmove', onEventTouchMove);
    }
});